/*
 * Copyright (C) 2012-2025 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.android.https.imagesearch;

import android.os.AsyncTask;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;

public abstract class HttpImageSearch extends AsyncTask<String, Void, Object> {

    @Override
    protected Object doInBackground(String... params) {
        HttpURLConnection con;
        byte[] responseArray = null;
        try {
            // --------------------------------------------------------
            // Communication: Obtain a image
            // --------------------------------------------------------
            // *** POINT 1 *** Sensitive information must not be contained in send
            // data.
            // Send image URL (after checking image_url)
            String image_url =
                "http://www.jssec.org/common/images/main_visual_local.jpg";
            con = connectUrl(image_url);
            checkResponse(con);

            // *** POINT 2 *** Suppose that received data may be sent from
            // attackers.
            // This is sample, so omit the process in case of the searching result
            // is the data from an attacker.
            responseArray = getByteArray(con);
            if (responseArray == null) {
                return null;
            }
        } catch (IOException e) {
            // Exception handling is omitted
        }
        return responseArray;
    }

    private HttpURLConnection connectUrl(String strUrl) {
        HttpURLConnection con = null;
        try {
            URL url = new URL(strUrl);
            con = (HttpURLConnection) url.openConnection();
            con.setRequestMethod("GET");
            con.connect();
        } catch (ProtocolException e) {
            // Handle exception (omitted)
        } catch (MalformedURLException e) {
            // Handle exception (omitted)
        } catch (IOException e) {
            // Handle exception (omitted)
        }
        return con;
    }

    private byte[] getByteArray(HttpURLConnection con) {
        byte[] buff = new byte[1024];
        byte[] result = null;
        BufferedInputStream inputStream = null;
        ByteArrayOutputStream responseArray = null;
        int length;

        try {
            inputStream = new BufferedInputStream(con.getInputStream());
            responseArray = new ByteArrayOutputStream();

            while ((length = inputStream.read(buff)) != -1) {
                if (length > 0) {
                    responseArray.write(buff, 0, length);
                }
            }
            result = responseArray.toByteArray();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                } catch (IOException e) {
                    // Exception handling is omitted
                }
            }
            if (responseArray != null) {
                try {
                    responseArray.close();
                } catch (IOException e) {
                    // Exception handling is omitted
                }
            }
        }
        return result;
    }

    private void checkResponse(HttpURLConnection response) throws IOException {
        int statusCode = response.getResponseCode();
        if (HttpURLConnection.HTTP_OK != statusCode) {
            throw new IOException("HttpStatus: " + statusCode);
        }
    }
}
